<?php

namespace App\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use AsocialMedia\Sfera\GT;

class ProductController extends Controller
{
    /**
     * Dodaje nowy produkt do Subiekta.
     *
     * @param  Request  $request
     *
     * @return JsonResponse
     * @api
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $this->validateProduct($request);

        try {
            if ($this->gt->Towary->Istnieje($validated['symbol'])) {
                abort(400, 'Produkt o podanym symbolu już istnieje');
            }

            $product = $this->gt->TowaryManager->DodajTowar();
            $this->saveProduct($product, $validated);
            $headers = [
                'Location' => route('products.show', [ 'symbol' => $product->Symbol ]),
            ];

            return $this->returnSuccess($product, 'Pomyślnie dodano nowy produkt', 201, $headers);
        } catch (Exception $e) {
            abort(400, $e->getMessage());
        }
    }

    /**
     * Waliduje poprawność przekazanych danych.
     *
     * @param  Request  $request
     * @param  string   $symbol  Symbol produktu służący do wyszukiwania w Subiekcie
     *
     * @return array
     */
    protected function validateProduct(Request $request, string $symbol = ''): array
    {
        $name = $request->json('nazwa');
        $symbol = $symbol ?: $request->json('symbol');
        $barcode = $request->json('kodKreskowy');
        $description = $request->json('opis');
        $price = filter_var($request->json('cena'), FILTER_VALIDATE_FLOAT);
        $features = $request->json('cechy') ?? [];
//        $availability = filter_var($request->json('stanMagazynowy') ?? 0, FILTER_VALIDATE_INT);
        $contractor = $request->json('kontrahent') ?? '';

        if (!$symbol) {
            abort(400, 'Symbol produktu jest wymagany');
        } else {
            $symbol = Str::limit($symbol, 20, '');
        }

        if (!$name) {
            abort(400, 'Nazwa produktu jest wymagana');
        } else {
            $name = Str::limit($name, 50, '');
        }

        if (!is_array($features)) {
            abort(400, 'Cechy są w niepoprawnym formacie');
        }

        if ($barcode) {
            $barcode = Str::limit($barcode, 20, '');
        }

        $ret = [
            'name' => $name,
            'symbol' => $symbol,
            'barcode' => $barcode,
//            'availability' => $availability,
        ];

        if ($price > 0) {
            $ret['price'] = $price;
        }

        if ($description) {
            $ret['description'] = $description;
        }

        if ($features) {
            $ret['features'] = $features;
        }

        if ($contractor) {
            $ret['contractor'] = $contractor;
        }

        return $ret;
    }

    /**
     * Zapisuje zmiany w produkcie w Subiekcie.
     * Działa zarówno przy dodawaniu nowego produktu, jak i przy aktualizacji.
     *
     * @param  GT     $product  Obiekt produktu w Subiekcie
     * @param  array  $data  Dane produktu do zapisu
     *
     * @return void
     */
    protected function saveProduct($product, array $data): void
    {
        $product->Nazwa = $data['name'];
        $product->Symbol = $data['symbol'];
        $product->DoSklepuInternetowego = true;

        if ($data['barcode']) {
            $product->KodyKreskowe->Podstawowy = $data['barcode'];
        }

        if (isset($data['description'])) {
            $product->Opis = $data['description'];
        }

        if (isset($data['price'])) {
            $product->Ceny->Element(1)->Brutto = $data['price'];
        }

        $product->Aktywny = true;
        $product->DoSklepuInternetowego = true;
        $product->Zapisz();

        if (isset($data['features'])) {
            $allFeatures = DB::table('sl_CechaTw')->get()->pluck('ctw_Nazwa', 'ctw_Id')->toArray();
            DB::table('tw_CechaTw')->where('cht_IdTowar', $product->Identyfikator)->delete();

            foreach ($data['features'] as $feature) {
                if (in_array($feature, $allFeatures)) {
                    $id = array_search($feature, $allFeatures);
                } else {
                    $id = DB::table('sl_CechaTw')->max('ctw_Id') + 1;
                    DB::table('sl_CechaTw')->insert([ 'ctw_Id' => $id, 'ctw_Nazwa' => $feature ]);
                }

                $featureObj = DB::table('sl_CechaTw')->where('ctw_Id', $id)->first();
                $product->Cechy->Dodaj($featureObj->ctw_Id);
                $product->Zapisz();
            }
        }

//        if (isset($data['availability']) && $data['availability'] > 0) {
//            $document = $this->gt->SuDokumentyManager->DodajPW();
//            $documentProduct = $document->Pozycje->Dodaj($data['symbol']);
//            $documentProduct->IloscJm = $data['availability'];
//            $documentProduct->Jm = 'szt.';
//
//            if (isset($data['contractor'])) {
//                $documentProduct->Tytuł = $data['contractor'];
//            }
//
//            $document->Zapisz();
//        }
    }

    /**
     * Zwraca odpowiedź w ustalonym formacie jeśli wszystkie operacje powiodły się.
     *
     * @param  GT      $product  Obiekt produktu w Subiekcie
     * @param  string  $message  Wiadomość umieszczana w zwrotce, opcjonalna
     * @param  int     $status  Kod HTTP odpowiedzi, domyślnie 200
     * @param  array   $headers  Nagłówki dołączane do odpowiedzi
     *
     * @return JsonResponse
     */
    protected function returnSuccess(
        $product,
        string $message = '',
        int $status = 200,
        array $headers = []
    ): JsonResponse {
        $features = [];
        $allFeatures = DB::table('sl_CechaTw')->get()->pluck('ctw_Nazwa', 'ctw_Id');
        $productFeatures = DB::table('tw_CechaTw')->where('cht_IdTowar', $product->Identyfikator)->get();

        foreach ($productFeatures as $key => $feature) {
            $features[] = $allFeatures[$feature->cht_IdCecha];
        }

        $ret = [
            'status' => 'success',
            'message' => $message,
            'data' => [
                'id' => $product->Identyfikator,
                'symbol' => $product->Symbol,
                'nazwa' => $product->Nazwa,
                'kodKreskowy' => $product->KodyKreskowe->Podstawowy,
                'opis' => $product->Opis,
                'cechy' => $features,
            ],
        ];

        if (!$message) {
            unset($ret['message']);
        }

        if ($product->Ceny->Liczba > 0) {
            $ret['data']['cena'] = (float) $product->Ceny->Element(1)->Brutto;
        }

        return response()->json($ret, $status)->withHeaders($headers);
    }

    /**
     * Pobiera produkt z Subiekta.
     *
     * @param  string  $symbol  Symbol produktu służący do wyszukiwania w Subiekcie
     *
     * @return JsonResponse
     * @api
     */
    public function show(string $symbol): JsonResponse
    {
        if (!$symbol || !$this->gt->Towary->Istnieje($symbol)) {
            abort(404, 'Nie istnieje produkt o podanym symbolu');
        }

        try {
            return $this->returnSuccess($this->gt->Towary->Wczytaj($symbol));
        } catch (Exception $e) {
            abort(400, $e->getMessage());
        }
    }

    /**
     * Aktualizuje produkt w Subiekcie.
     *
     * @param  Request  $request
     * @param  string   $symbol  Symbol produktu służący do wyszukiwania w Subiekcie
     *
     * @return JsonResponse
     * @api
     */
    public function update(Request $request, string $symbol): JsonResponse
    {
        $validated = $this->validateProduct($request, $symbol);

        if (!$this->gt->Towary->Istnieje($validated['symbol'])) {
            abort(404, 'Nie istnieje produkt o podanym symbolu');
        }

        try {
            $product = $this->gt->Towary->Wczytaj($validated['symbol']);
            $this->saveProduct($product, $validated);

            return $this->returnSuccess($product, 'Pomyślnie zaktualizowano produkt');
        } catch (Exception $e) {
            abort(400, $e->getMessage());
        }
    }
}
