<?php

namespace App\Http\Controllers;

use Exception;
use Illuminate\Database\QueryException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use AsocialMedia\Sfera\GT;

class OrderController extends Controller
{
    /**
     * Dodaje nowe zamówienie do Subiekta.
     *
     * @param  Request  $request
     *
     * @return JsonResponse
     * @api
     */
    public function store(Request $request): JsonResponse
    {
        $this->log('--------------------------------------------------');
        $this->log('Próba dodania ZK');
        $validated = $this->validateOrder($request);
        $this->log('Po walidacji danych, mogę dodać ZK');

        try {
            $warehouse = config('project.warehouse', null);

            if (!empty($warehouse)) {
                $this->log('Ustawiam magazyn ' . $warehouse);
                $this->gt->MagazynId = (int) $warehouse;
            }

            $this->log('Tworzę nowe ZK');
            $order = $this->gt->SuDokumentyManager->DodajZK();
            $this->log('Zapisuję dane w ZK');
            $data = $this->saveOrder($order, $validated);
            $this->log('Dane zapisane, udało się stworzyć ZK');

            return $this->returnSuccess($data, 'Pomyślnie dodano nowe zamówienie', 201);
        } catch (Exception $e) {
            $this->log('Wystąpił błąd tworzenia ZK ' . $e->getMessage());
            abort(400, $e->getMessage());
        }
    }

    /**
     * Wystawia dokument sprzedaży (fakturę lub paragon imienny) na podstawie zamówienia klienta.
     *
     * @param  Request  $request
     *
     * @return JsonResponse
     */
    public function execute(Request $request): JsonResponse
    {
        try {
            if ($request->input('faktura')) {
                $faktura = $this->gt->SuDokumentyManager->DodajFS();
            } else {
                $faktura = $this->gt->SuDokumentyManager->DodajPAi();
                $faktura->RejestrujNaUF = true;
            }

            $document = $this->getOrderByOriginalNumber($request->input('zamowienie'));
            $faktura->NaPodstawie($document->dok_Id);
//        $faktura->Podtytul = $order->Tytul;
            $faktura->Zapisz();

            return $this->returnSuccess([ 'nr' => $faktura->NumerPelny ], 'Pomyślnie wystawiono dokument zamówienia');
        } catch (Exception $e) {
            abort(400, $e->getMessage());
        }
    }

    /**
     * Ustawia lub usuwa flagę "Do uszykowania" w zależności od przekazanego statusu.
     *
     * @param  Request  $request
     * @param  string   $id
     *
     * @return JsonResponse
     */
    public function status(Request $request, string $id): JsonResponse
    {
        $this->log('--------------------------------------------------');
        $this->log('Ustawianie statusu dla dokumentu ' . $id);
        $document = $this->getOrderByOriginalNumber($id);
        $this->log('Pobrałem dokument po nr. oryg., id  ' . $document->dok_NrPelny);
        $order = $this->gt->SuDokumentyManager->Wczytaj($document->dok_NrPelny);
        $this->log('Wczytałem dokument, zmieniam flagę');

        if ($request->input('doUszykowania')) {
            $order->FlagaNazwa = 'Do uszykowania';
        } else {
            $order->FlagaNazwa = '';
        }

        $this->log('Flaga zmieniona, zapisuję dokument');
        $order->Zapisz();
        $this->log('Pomyślnie zapisano dokument ' . $document->dok_Id);

        return $this->returnSuccess([ 'nr' => $document->dok_Id ], 'Pomyślnie zaktualizowano status zamówienia');
    }

    /**
     * Zmiana sposobu płatności ZK na pobranie.
     *
     * @param  Request  $request
     * @param  string   $id
     *
     * @return JsonResponse
     */
    public function cod(Request $request, string $id): JsonResponse
    {
        $document = $this->getOrderByOriginalNumber($id);
        $order = $this->gt->SuDokumentyManager->Wczytaj($document->dok_NrPelny);

        $orderService = $order->Pozycje->DodajUslugeJednorazowa();
        $orderService->UslJednNazwa = 'Płatność za pobraniem';
        $orderService->WartoscBruttoPoRabacie = (float) $request->input('kwota');

        $order->PlatnoscKartaId = 13;
        $order->PlatnoscKartaKwota = $order->KwotaDoZaplaty;
        $order->PoleWlasne['KURIER'] = 21;
        $order->Przelicz();
        $order->Zapisz();

        $order->FlagaNazwa = 'Do uszykowania';
        $order->Zapisz();

        return $this->returnSuccess([ 'nr' => $document->dok_Id ],
            'Pomyślnie zaktualizowano sposób płatności zamówienia');
    }

    /**
     * Anuluje ZK w Subiekcie.
     *
     * @param  Request  $request
     * @param  string   $id
     *
     * @return JsonResponse
     * @api
     */
    public function cancel(Request $request, string $id): JsonResponse
    {
        try {
            $this->log('--------------------------------------------------');
            $this->log('Próbuję usunąć ZK ' . $id);
            $document = $this->getOrderByOriginalNumber($id);
            $this->log('Znalazłem ZK po numerze oryinalnym');
            $docId = $document->dok_Id;
            $order = $this->gt->SuDokumentyManager->Wczytaj($document->dok_NrPelny);
            $this->log('Dokument wczytany');

            if ($order->PoleWlasne['Arvato_status'] == 0) {
                $this->log('Arvato status 0, usuwam');
                $order->Usun();
                $this->log('Dokument usunięty');

                return $this->returnSuccess([ 'nr' => $docId ], 'Pomyślnie usunięto zamówienie');
            }

            $this->log('Nie zgadza się status Arvato ' . $order->PoleWlasne['Arvato_status']);

            return $this->returnSuccess([ 'nr' => $docId ],
                'Nie usunięto zamówienia ze względu na status Arvato: ' . $order->PoleWlasne['Arvato_status']);
        } catch (Exception $e) {
            $this->log('Wystąpił błąd usuwania ZK ' . $e->getMessage());
            abort(400, $e->getMessage());
        }
    }

    /**
     * Waliduje poprawność przekazanych danych.
     *
     * @param  Request  $request
     *
     * @return array
     */
    protected function validateOrder(Request $request): array
    {
        $contractor = $request->input('kontrahent');
        $deliveryAddress = $request->input('adresDostawy');
        $products = $request->input('produkty');
        $services = $request->input('uslugi');
        $reference = $request->input('numer');
        $comments = $request->input('uwagi');
        $amount = $request->input('kwota');
        $payment = $request->input('platnoscId');
        $payed = filter_var($request->input('oplacone'), FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        //$arvato = filter_var($request->input('arvato'), FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        $prepare = filter_var($request->input('doUszykowania'), FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        $courier = $request->input('kurier');
        $isInvoice = filter_var($request->input('czy_faktura'), FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);

        if (!$contractor) {
            abort(400, 'Kontrahent jest wymagany');
        } elseif (!is_array($contractor) || !isset($contractor['email'])) {
            abort(400, 'Kontrahent jest w niepoprawnym formacie');
        } elseif ($deliveryAddress && !is_array($deliveryAddress)) {
            abort(400, 'Adres dostawy jest w niepoprawnym formacie');
        } elseif (!$products) {
            abort(400, 'Lista produktów jest wymagana');
        } elseif (!is_array($products)) {
            abort(400, 'Lista produktów jest w niepoprawnym formacie');
        } elseif (!$reference) {
            abort(400, 'Numer zamówienia jest wymagany');
//        } elseif (!$amount) {
//            abort(400, 'Kwota zamówienia jest wymagana');
        }

        foreach ($products as $productId => $product) {
            if (!$this->gt->Towary->Istnieje($productId)) {
                abort(404, 'Nie istnieje produkt o podanym identyfikatorze: ' . $productId);
            }

            $products[$productId]['obiekt'] = $this->gt->Towary->Wczytaj($productId);
        }

        $ret = [
            'products' => $products,
            'services' => $services,
            'reference' => $reference,
            'payment' => $payment,
            'payed' => $payed,
            //'arvato' => $arvato,
            'prepare' => $prepare,
            'courier' => $courier,
            'isInvoice' => $isInvoice,
            'contractor' => $contractor,
        ];

        /*$ret['contractor'] = $this->getContractorByEmail($contractor['email']);

        if (!$ret['contractor']) {
            $ret['tempContractor'] = $contractor;
        }*/

        if ($deliveryAddress) {
            $ret['deliveryAddress'] = $deliveryAddress;
        }

        if ($comments) {
            $ret['comments'] = $comments;
        }

        if ($amount) {
            $ret['amount'] = $amount;
        }

        return $ret;
    }

    /**
     * Zapisuje zamówienie w Subiekcie.
     *
     * @param  GT     $order
     * @param  array  $data
     *
     * @return array
     */
    protected function saveOrder($order, array $data): array
    {
        $symbol = Str::substr($data['contractor']['email'], 0, 20);
        $order->Tytul = $data['reference'];
        $order->Rezerwacja = true;
        $order->LiczonyOdCenBrutto = true;
        $order->PoziomCenyId = 1;
        $deliveryAddressId = null;
        $this->log('Podstawowe dane ustawione');

        if ($this->gt->KontrahenciManager->Istnieje($symbol)) {
            $this->log('Wczytuję kontrahenta o symbolu ' . $symbol);
            $contractor = $this->gt->KontrahenciManager->Wczytaj($symbol);
            $this->log('Kontrahent wczytany');
            $name = $this->getContractorData($data['contractor'],
                'nazwa') !== '' ? $this->getContractorData($data['contractor'],
                'nazwa') : $this->getContractorData($data['contractor'],
                    'imie') . ' ' . $this->getContractorData($data['contractor'], 'nazwisko');
            $street = Str::substr($this->getContractorData($data['contractor'], 'ulica'), 0, 60);
            $houseNo = Str::substr($this->getContractorData($data['contractor'], 'nrDomu'), 0, 10);
            $apartmentNo = Str::substr($this->getContractorData($data['contractor'], 'nrLokalu'), 0, 10);
            $zipCode = Str::substr($this->getContractorData($data['contractor'], 'kodPocztowy'), 0, 8);
            $city = Str::substr($this->getContractorData($data['contractor'], 'miejscowosc'), 0, 40);
            $phone = Str::substr($this->getContractorData($data['contractor'], 'telefon'), 0, 35);

            if (isset($data['deliveryAddress'])) {
                $this->log('Przesłano adres dostawy');
                $deliveryStreet = Str::substr($this->getContractorData($data['deliveryAddress'], 'ulica'), 0, 60);
                $deliveryHouseNo = Str::substr($this->getContractorData($data['deliveryAddress'], 'nrDomu'), 0, 10);
                $deliveryApartmentNo = Str::substr($this->getContractorData($data['deliveryAddress'], 'nrLokalu'), 0,
                    10);
                $deliveryZipCode = Str::substr($this->getContractorData($data['deliveryAddress'], 'kodPocztowy'), 0, 8);
                $deliveryCity = Str::substr($this->getContractorData($data['deliveryAddress'], 'miejscowosc'), 0, 40);
                $deliveryPhone = Str::substr($this->getContractorData($data['deliveryAddress'], 'telefon'), 0, 35);

                if ($contractor->AdresyDostaw->Liczba() > 0) {
                    $this->log('Kontrahent ma już adresy dostaw');

                    for ($i = 1; $i <= $contractor->AdresyDostaw->Liczba(); $i++) {
                        $address = $contractor->AdresyDostaw->Element($i);

                        if ($address->Miejscowosc === $deliveryCity
                            && $address->KodPocztowy === $deliveryZipCode
                            && $address->Ulica === $deliveryStreet
                            && $address->NrDomu === $deliveryHouseNo
                            && $address->NrLokalu === $deliveryApartmentNo
                            && $address->Telefon === $deliveryPhone
                        ) {
                            $deliveryAddressId = $this->findDeliveryAddressId($contractor->Identyfikator, $name,
                                $deliveryStreet, $deliveryHouseNo, $deliveryApartmentNo, $deliveryZipCode,
                                $deliveryCity,
                                $deliveryPhone);
                            break;
                        }
                    }

                    if (!$deliveryAddressId) {
                        $this->log('Nie znalazłem zgodnego adresu dostawy, dodaję nowy');
                        $address = $contractor->AdresyDostaw->Dodaj($contractor->Identyfikator);
                        $address->Nazwa = $name;
                        $address->Ulica = $deliveryStreet;
                        $address->NrDomu = $deliveryHouseNo;
                        $address->NrLokalu = $deliveryApartmentNo;
                        $address->KodPocztowy = $deliveryZipCode;
                        $address->Miejscowosc = $deliveryCity;
                        $address->Panstwo = $this->getCountry($this->getContractorData($data['deliveryAddress'],
                            'kraj'));
                        $address->Telefon = $deliveryPhone;
                        $contractor->AdresyDostaw->Zapisz();
                        $this->log('Adres dostawy zapisany');
                        $deliveryAddressId = $this->findDeliveryAddressId($contractor->Identyfikator, $name,
                            $deliveryStreet,
                            $deliveryHouseNo, $deliveryApartmentNo, $deliveryZipCode, $deliveryCity, $deliveryPhone);
                    }
                } else {
                    $this->log('Kontrahent nie ma adresów dostaw, dodaję nowy');
                    $address = $contractor->AdresyDostaw->Dodaj($contractor->Identyfikator);
                    $address->Nazwa = $name;
                    $address->Ulica = $deliveryStreet;
                    $address->NrDomu = $deliveryHouseNo;
                    $address->NrLokalu = $deliveryApartmentNo;
                    $address->KodPocztowy = $deliveryZipCode;
                    $address->Miejscowosc = $deliveryCity;
                    $address->Panstwo = $this->getCountry($this->getContractorData($data['deliveryAddress'], 'kraj'));
                    $address->Telefon = $deliveryPhone;
                    $contractor->AdresyDostaw->Zapisz();
                    $this->log('Adres dostawy zapisany');
                    $deliveryAddressId = $this->findDeliveryAddressId($contractor->Identyfikator, $name,
                        $deliveryStreet,
                        $deliveryHouseNo, $deliveryApartmentNo, $deliveryZipCode, $deliveryCity, $deliveryPhone);
                }
            } else/*if ($contractor->Miejscowosc !== $city
                || $contractor->KodPocztowy !== $zipCode
                || $contractor->Ulica !== $street
                || $contractor->NrDomu !== $houseNo
                || $contractor->NrLokalu !== $apartmentNo
            )*/ {
                if ($contractor->AdresyDostaw->Liczba() > 0) {
                    $this->log('Kontrahent ma już adresy dostaw');

                    for ($i = 1; $i <= $contractor->AdresyDostaw->Liczba(); $i++) {
                        $address = $contractor->AdresyDostaw->Element($i);

                        if ($address->Miejscowosc === $city
                            && $address->KodPocztowy === $zipCode
                            && $address->Ulica === $street
                            && $address->NrDomu === $houseNo
                            && $address->NrLokalu === $apartmentNo
                            && $address->Telefon === $phone
                        ) {
                            $deliveryAddressId = $this->findDeliveryAddressId($contractor->Identyfikator, $name,
                                $street,
                                $houseNo, $apartmentNo, $zipCode, $city, $phone);
                            break;
                        }
                    }

                    if (!$deliveryAddressId) {
                        $this->log('Nie znalazłem zgodnego adresu dostawy, dodaję nowy');
                        $address = $contractor->AdresyDostaw->Dodaj($contractor->Identyfikator);
                        $address->Nazwa = $name;
                        $address->Ulica = $street;
                        $address->NrDomu = $houseNo;
                        $address->NrLokalu = $apartmentNo;
                        $address->KodPocztowy = $zipCode;
                        $address->Miejscowosc = $city;
                        $address->Panstwo = $this->getCountry($this->getContractorData($data['contractor'], 'kraj'));
                        $address->Telefon = $phone;
                        $contractor->AdresyDostaw->Zapisz();
                        $this->log('Adres dostawy zapisany');
                        $deliveryAddressId = $this->findDeliveryAddressId($contractor->Identyfikator, $name, $street,
                            $houseNo, $apartmentNo, $zipCode, $city, $phone);
                    }
                } else {
                    $this->log('Kontrahent nie ma adresów dostaw, dodaję nowy');
                    $address = $contractor->AdresyDostaw->Dodaj($contractor->Identyfikator);
                    $address->Nazwa = $name;
                    $address->Ulica = $street;
                    $address->NrDomu = $houseNo;
                    $address->NrLokalu = $apartmentNo;
                    $address->KodPocztowy = $zipCode;
                    $address->Miejscowosc = $city;
                    $address->Panstwo = $this->getCountry($this->getContractorData($data['contractor'], 'kraj'));
                    $address->Telefon = $phone;
                    $contractor->AdresyDostaw->Zapisz();
                    $this->log('Adres dostawy zapisany');
                    $deliveryAddressId = $this->findDeliveryAddressId($contractor->Identyfikator, $name, $street,
                        $houseNo, $apartmentNo, $zipCode, $city, $phone);
                }
            }
        } elseif (isset($data['tempContractor'])) {
            $this->log('Dodaję kontrahenta jednorazowego');
            $contractor = $this->gt->KontrahenciManager->DodajKontrahentaJednorazowego();

            if ($data['tempContractor']['czy_firma']) {
                $contractor->NazwaPelna = $this->getContractorData($data['tempContractor'], 'nazwa');
                $contractor->Nazwa = Str::substr($this->getContractorData($data['tempContractor'], 'nazwa'), 0, 100);
                $contractor->NIP = $this->getContractorData($data['tempContractor'], 'nip');
            } else {
                $nazwa = $this->getContractorData($data['tempContractor'],
                        'imie') . ' ' . $this->getContractorData($data['tempContractor'], 'nazwisko');
                $contractor->NazwaPelna = $nazwa;
                $contractor->Nazwa = Str::substr($nazwa, 0, 100);
            }

            $contractor->Email = $this->getContractorData($data['tempContractor'], 'email');
            $contractor->Miejscowosc = $this->getContractorData($data['tempContractor'], 'miejscowosc');
            $contractor->KodPocztowy = Str::substr($this->getContractorData($data['tempContractor'], 'kodPocztowy'), 0,
                8);
            $contractor->Ulica = Str::substr($this->getContractorData($data['tempContractor'], 'ulica'), 0, 60);
            $contractor->NrDomu = Str::substr($this->getContractorData($data['tempContractor'], 'nrDomu'), 0, 10);
            $contractor->NrLokalu = $this->getContractorData($data['tempContractor'], 'nrLokalu');
            $contractor->Panstwo = $this->getCountry($this->getContractorData($data['tempContractor'], 'kraj'));

            if (isset($data['tempContractor']['telefon'])) {
                $phone = $contractor->Telefony->Dodaj($data['tempContractor']['telefon']);
                $phone->Nazwa = 'Numer kontaktowy';
                $phone->Numer = $data['tempContractor']['telefon'];
                $phone->Typ = 3;
            }

            $contractor->Zapisz();
        } else {
            $this->log('Dodaję nowego kontrahenta o symbolu ' . $symbol);
            $contractor = $this->gt->KontrahenciManager->DodajKontrahenta();
            $contractor->Symbol = $symbol;

            $name = $this->getContractorData($data['contractor'],
                'nazwa') !== '' ? $this->getContractorData($data['contractor'],
                'nazwa') : $this->getContractorData($data['contractor'],
                    'imie') . ' ' . $this->getContractorData($data['contractor'], 'nazwisko');
            $street = Str::substr($this->getContractorData($data['contractor'], 'ulica'), 0, 60);
            $houseNo = Str::substr($this->getContractorData($data['contractor'], 'nrDomu'), 0, 10);
            $apartmentNo = Str::substr($this->getContractorData($data['contractor'], 'nrLokalu'), 0, 10);
            $zipCode = Str::substr($this->getContractorData($data['contractor'], 'kodPocztowy'), 0, 8);
            $city = Str::substr($this->getContractorData($data['contractor'], 'miejscowosc'), 0, 40);
            $phone = Str::substr($this->getContractorData($data['contractor'], 'telefon'), 0, 35);

            if ($data['contractor']['czy_firma']) {
                $this->log('Kontrahent jest firmą');
                $contractor->Osoba = 0;
                $contractor->NazwaPelna = $this->getContractorData($data['contractor'], 'nazwa');
                $contractor->Nazwa = Str::substr($this->getContractorData($data['contractor'], 'nazwa'), 0, 100);
                $contractor->NIP = $this->getContractorData($data['contractor'], 'nip');
                $contractor->PowielNIPBezUI = true;
            } else {
                $this->log('Kontrahent jest osobą fizyczną');
                $contractor->Osoba = 1;
                $contractor->NazwaPelna = $this->getContractorData($data['contractor'],
                        'imie') . ' ' . $this->getContractorData($data['contractor'], 'nazwisko');
                $contractor->OsobaImie = Str::substr($this->getContractorData($data['contractor'], 'imie'), 0, 20);
                $contractor->OsobaNazwisko = Str::substr($this->getContractorData($data['contractor'], 'nazwisko'), 0,
                    50);
            }

            $this->log('Zapisuję podstawowe dane kontrahenta');
            $contractor->Email = $data['contractor']['email'];
            $contractor->Miejscowosc = $city;
            $contractor->KodPocztowy = $zipCode;
            $contractor->Ulica = $street;
            $contractor->NrDomu = $houseNo;
            $contractor->NrLokalu = $apartmentNo;
            $contractor->Panstwo = $this->getCountry($this->getContractorData($data['contractor'], 'kraj'));

            if (!empty($data['contractor']['telefon'])) {
                $this->log('Kontrahent ma numer telefonu');
                $phone = $contractor->Telefony->Dodaj($phone);
                $phone->Nazwa = 'Numer kontaktowy';
                $phone->Numer = $data['contractor']['telefon'];
                $phone->Typ = 3;
            }

            $this->log('Zapisuję kontrahenta');
            $contractor->Zapisz();

            $this->log('Kontrahent zapisany, dodaję adres dostawy');
            $address = $contractor->AdresyDostaw->Dodaj($contractor->Identyfikator);
            $address->Nazwa = $name;
            $address->Ulica = $street;
            $address->NrDomu = $houseNo;
            $address->NrLokalu = $apartmentNo;
            $address->KodPocztowy = $zipCode;
            $address->Miejscowosc = $city;
            $address->Panstwo = $this->getCountry($this->getContractorData($data['contractor'], 'kraj'));
            $address->Telefon = $phone;
            $contractor->AdresyDostaw->Zapisz();
            $this->log('Adres dostawy zapisany');
            $deliveryAddressId = $this->findDeliveryAddressId($contractor->Identyfikator, $name, $street,
                $houseNo, $apartmentNo, $zipCode, $city, $phone);
        }

        $this->log('Przypisuję do zamówienia kontrahenta ' . $contractor->Identyfikator);
        $order->KontrahentId = $contractor->Identyfikator;

        if ($deliveryAddressId) {
            $this->log('Przypisuję do zamówienia adres dostawy ' . $deliveryAddressId);
            $order->AdresDostawyId = $deliveryAddressId;
        }

        $this->log('Zaczynam dodawać produkty do ZK');

        foreach ($data['products'] as $product) {
            $orderProduct = $order->Pozycje->Dodaj($product['obiekt']->Symbol);
            $orderProduct->IloscJm = $product['ilosc'];
            $orderProduct->RabatProcent = $product['rabat'];
            $orderProduct->WartoscBruttoPoRabacie = $product['ilosc'] * $product['cena'];
        }

        $this->log('Zaczynam dodawać usługi do ZK');

        if (isset($data['services'])) {
            foreach ($data['services'] as $service => $price) {
                $orderService = $order->Pozycje->DodajUslugeJednorazowa();
                $orderService->UslJednNazwa = $service;
                $orderService->WartoscBruttoPoRabacie = $price;
            }
        }

        $this->log('Dodaję numer oryginalny oraz dane płatności');
        $order->NumerOryginalny = $data['reference'];
        $order->PlatnoscKartaKwota = $order->KwotaDoZaplaty;
        $order->PlatnoscKartaId = (int) $data['payment'];
        $this->log('Ustawiam pola własne');
        //$order->PoleWlasne['Arvato_status'] = (int) $data['arvato'];
        $order->PoleWlasne['Czy faktura?'] = (int) $data['isInvoice'];

        if ($data['courier']) {
            $order->PoleWlasne['KURIER'] = $data['courier'];
        }

        if (isset($data['comments'])) {
            $this->log('Dodaję uwagi');
            $order->Uwagi = $data['comments'];
        }

        $this->log('Przeliczam i zapisuję ZK');
        $order->Przelicz();
        $order->Zapisz();
        $this->log('ZK zapisane');

        if ($data['prepare']) {
            $this->log('Dodaję flagę "do uszykowania:');
            $order->FlagaNazwa = 'Do uszykowania';
            $order->Zapisz();
            $this->log('Flaga zapisana');
        }

        $this->log('Wszystko ok, ZK ma numer ' . $order->Identyfikator);

        return [
            'id' => $order->Identyfikator,
            'numerZamowienia' => $order->NumerPelny,
            'numerOryginalny' => $order->Tytul,
        ];
    }

    /**
     * Pobiera przesłane dane kontrahenta dla podanego klucza.
     *
     * @param  array   $data  Tablica z danymi
     * @param  string  $key  Nazwa klucza, dla którego pobieramy dane
     *
     * @return string
     */
    protected function getContractorData(array $data, string $key): string
    {
        if (!$data) {
            return '';
        }

        return $data[$key] ?? '';
    }

    /**
     * Pobiera identyfikator kraju dla podanej nazwy.
     *
     * @param  string  $name
     *
     * @return int
     */
    protected function getCountry(string $name): int
    {
        if (empty($name)) {
            return 1; // Polska
//            return -2147483648; // Brak
        }

        $country = DB::table('sl_Panstwo')->where('pa_Nazwa', $name)->first();

        if (!$country) {
            return 1; // Polska
//            return -2147483648; // Brak
        }

        return $country->pa_Id;
    }

    /**
     * Pobiera z bazy identyfikator pasującego adresu dostawy.
     *
     * @param  int     $contractorId
     * @param  string  $name
     * @param  string  $street
     * @param  string  $houseNo
     * @param  string  $apartmentNo
     * @param  string  $zipCode
     * @param  string  $city
     * @param  string  $phone
     *
     * @return int|null
     */
    protected function findDeliveryAddressId(
        $contractorId,
        $name,
        $street,
        $houseNo,
        $apartmentNo,
        $zipCode,
        $city,
        $phone
    ) {
        $deliveryAddress = DB::table('adr_Historia')
            ->join('kh_AdresyDostawy', 'adk_IdAdresuHist', '=', 'adrh_Id')
            ->where('adk_IdKhnt', $contractorId)
            ->where('adrh_Nazwa', $name)
            ->where('adrh_Ulica', $street)
            ->where('adrh_NrDomu', $houseNo)
            ->where('adrh_NrLokalu', $apartmentNo)
            ->where('adrh_Kod', $zipCode)
            ->where('adrh_Miejscowosc', $city)
            ->where('adrh_Telefon', $phone)
            ->first();

        if (!$deliveryAddress) {
            return null;
        }

        return $deliveryAddress->adrh_Id;
    }

    /**
     * Zwraca odpowiedź w ustalonym formacie jeśli wszystkie operacje powiodły się.
     *
     * @param  array   $data  Obiekt zamówienia w Subiekcie
     * @param  string  $message  Wiadomość umieszczana w zwrotce, opcjonalna
     * @param  int     $status  Kod HTTP odpowiedzi, domyślnie 200
     * @param  array   $headers  Nagłówki dołączane do odpowiedzi
     *
     * @return JsonResponse
     */
    protected function returnSuccess(
        $data,
        string $message = '',
        int $status = 200,
        array $headers = []
    ): JsonResponse {
        $ret = [
            'status' => 'success',
            'message' => $message,
            'data' => $data,
        ];

        if (!$message) {
            unset($ret['message']);
        }

        return response()->json($ret, $status)->withHeaders($headers);
    }

    /**
     * Pobiera ID zamówienia w Subiekcie na podstawie numeru zamówienia w sklepie.
     *
     * @param  string  $title
     *
     * @return \Illuminate\Database\Eloquent\Model
     */
    protected function getOrderByOriginalNumber(string $title)
    {
        $this->log('Szukam w bazie dokumentu po numerze oryginalnym ' . $title);
        $document = DB::table('vwDok4ZamGrid')->where('dok_NrPelnyOryg', $title)->first();

        if (!$document) {
            $this->log('Dokument nie istnieje');
            abort(404, 'Nie istnieje dokument o podanym numerze oryginalnym: ' . $title);
        }

        $this->log('Znalazłem dokument');

        return $document;
    }

    /**
     * Pobiera produkt na podstawie ID (Sfera domyślnie pozwala wyszukiwać po symbolu).
     *
     * @param  int  $id
     *
     * @return GT|null
     */
    protected function getProductById(int $id)
    {
        try {
            return $this->gt->Towary[$id];
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * Pobiera kontrahenta na podstawie ID (Sfera domyślnie pozwala wyszukiwać po symbolu).
     *
     * @param  int  $id
     *
     * @return GT|null
     */
    protected function getContractorById(int $id)
    {
        try {
            return $this->gt->Kontrahenci[$id];
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * Pobiera kontrahenta na podstawie adresu e-mail (Sfera domyślnie pozwala wyszukiwać po symbolu).
     *
     * @param  string  $email
     *
     * @return GT|null
     */
    protected function getContractorByEmail(string $email)
    {
        if ($email == '') {
            return null;
        }

        try {
            $sql = DB::select('SELECT TOP (1) kh_Symbol AS symbol FROM kh__Kontrahent WHERE kh_Rodzaj = 2 AND kh_EMail = ?',
                [ $email ]);
        } catch (QueryException $e) {
            return null;
        }

        try {
            return $this->gt->KontrahenciManager->Wczytaj($sql[0]->symbol);
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * Sprawdza czy w Subiekcie istnieje ZK dla podanego zamówienia
     *
     * @param  Request  $request
     * @param  string   $id
     *
     * @return JsonResponse
     */
    public function show(Request $request, string $id): JsonResponse
    {
        $document = $this->getOrderByOriginalNumber($id);
//        $order = $this->gt->SuDokumentyManager->Wczytaj($document->dok_NrPelny);
        $ret = [
            'nr' => $document->dok_Id,
        ];

        return $this->returnSuccess($ret, 'Pomyślnie pobrano ZK');
    }

    /**
     * Wyświetla logi synchronizacji
     *
     * @return string
     */
    public function showlogs(Request $request): string
    {
        return file_get_contents(storage_path('synchronizacja.txt'));
    }

    /**
     * Wyświetla logi z przejścia programu
     *
     * @return string
     */
    public function runtimelogs(): string
    {
        return file_get_contents(storage_path('logi.txt'));
    }

    /**
     * @return string
     */
    public function pinglogs(): string
    {
        return file_get_contents(storage_path('ping.txt'));
    }

    /**
     * Loguje wiadomość do pliku wraz z datą
     *
     * @param  string  $message
     */
    protected function log(string $message): void
    {
        file_put_contents(storage_path('logi.txt'), date('Y-m-d H:i:s.v') . ' ' . $message . PHP_EOL, FILE_APPEND);
    }
}
