<?php

namespace App\Exceptions;

use Exception;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Validation\ValidationException;
use Laravel\Lumen\Exceptions\Handler as ExceptionHandler;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\HttpKernel\Exception\MethodNotAllowedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class Handler extends ExceptionHandler
{
    /**
     * A list of the exception types that should not be reported.
     *
     * @var array
     */
    protected $dontReport = [
        AuthorizationException::class,
        HttpException::class,
        ModelNotFoundException::class,
        ValidationException::class,
    ];

    /**
     * Report or log an exception.
     *
     * This is a great spot to send exceptions to Sentry, Bugsnag, etc.
     *
     * @param  \Exception  $exception
     *
     * @return void
     * @throws Exception
     */
    public function report(Exception $exception)
    {
        parent::report($exception);
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Exception                $exception
     *
     * @return \Illuminate\Http\Response|\Illuminate\Http\JsonResponse
     */
    public function render($request, Exception $exception)
    {
        if ($request->wantsJson()) {
            $class = get_class($exception);
            $headers = [
                'Content-Type' => 'application/json; charset=UTF-8',
                'charset' => 'utf-8',
            ];

            switch ($class) {
                case NotFoundHttpException::class:
                case ModelNotFoundException::class:
                    return response()->json([
                        'status' => 'error',
                        'error' => [
                            'code' => 404,
                            'message' => $this->transformOutput($exception->getMessage()) ?: 'Not Found',
                        ],
                    ], 404, $headers, JSON_UNESCAPED_UNICODE);
                case MethodNotAllowedHttpException::class:
                    return response()->json([
                        'status' => 'error',
                        'error' => [
                            'code' => 405,
                            'message' => 'Method Not Allowed',
                        ],
                    ], 405, $headers, JSON_UNESCAPED_UNICODE);
                case BadRequestHttpException::class:
                case HttpException::class:
                    return response()->json([
                        'status' => 'error',
                        'error' => [
                            'code' => 400,
                            'message' => $this->transformOutput($exception->getMessage()) ?: 'Bad Request',
                        ],
                    ], 400, $headers, JSON_UNESCAPED_UNICODE);
            }
        }

        return parent::render($request, $exception);
    }

    /**
     * Transforms output by replacing new line characters and XHTML style br into HTML br.
     *
     * @param  string  $str
     *
     * @return string
     */
    protected function transformOutput($str)
    {
        return str_replace([ "\r\n", "\r", "\n", '<br/>', '<br />' ], '<br>', $str);
    }
}
