<?php

namespace App\Console\Commands;

use Log;
use Exception;
use GuzzleHttp\Client;
use Illuminate\Console\Command;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;

class SynchronizeInventory extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'synchronize-inventory';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Synchronizuje stany magazynowe z Subiekta z zewnętrznym sklepem';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        Log::info('[START] ' . $this->description);
        $this->log('info', '[START]' . $this->description);

        try {
            $warehouse = config('project.warehouse', null);
            $inventory = DB::table('tw__Towar')
                ->join('tw_Stan', 'st_TowId', '=', 'tw_Id')
                ->when($warehouse, function ($q) use ($warehouse) {
                    /** @var \Illuminate\Database\Query\Builder $q */
                    return $q->where('st_MagId', $warehouse);
                })
                ->selectRaw('tw_Symbol AS symbol, (SUM(st_stan) - SUM(st_stanRez)) AS stan')
                ->groupBy('tw_Symbol')
                ->get();
            $this->log('info', '[warehouse]' . print_r($warehouse, true));
            $this->log('info', '[inventory]' . print_r($inventory->toArray(), true));
        } catch (QueryException $e) {
            $inventory = false;
        }

        if (!$inventory) {
            $this->log('error', 'Nie udało się pobrać stanów magazynowych z bazy.');

            return;
        }

        try {
            $client = new Client();
            $response = $client->post(config('project.shop.url') . '?token=' . config('project.shop.token'), [
                'json' => $inventory->toArray(),
                'connect_timeout' => 300,
                'timeout' => 300,
            ]);
            $this->log('success',
                'Stany magazynowe wysłane: ' . count($inventory->toArray()) . ', sklep zwrócił status: ' . $response->getBody()->getContents());
            $this->log('info', '[inventory]' . print_r($inventory->toArray(), true));
        } catch (Exception $e) {
            $this->log('error',
                'Nie udało się wysłać stanów magazynowych: ' . count($inventory->toArray()) . ', sklep zwrócił błąd: ' . $e->getMessage());
        }

        Log::info('[END] ' . $this->description);
        $this->log('info', '[END]' . $this->description);
    }

    protected function log($status, $message)
    {
        $status = strtoupper($status);

        if (!in_array($status, [ 'SUCCESS', 'ERROR', 'WARNING', 'INFO' ])) {
            $status = 'INFO';
        }

        $log = implode(' ', [
                date('Y-m-d H:i:s'),
                $status,
                $message,
            ]) . PHP_EOL;

        file_put_contents(storage_path($this->signature . '.txt'), $log, FILE_APPEND);
    }
}
