<?php

namespace App\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Zoondo\Sfera\GT;

class ContractorController extends Controller
{
    /**
     * Dodaje nowego kontrahenta do Subiekta.
     *
     * @param  Request  $request
     *
     * @return JsonResponse
     * @api
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $this->validateContractor($request);

        try {
            if ($this->gt->Kontrahenci->Istnieje($validated['symbol'])) {
                abort(400, 'Kontrahent o podanym symbolu już istnieje');
            }

            $contractor = $this->gt->Kontrahenci->Dodaj();
            $this->saveContractor($contractor, $validated);
            $headers = [
                'Location' => route('contractors.show', [ 'symbol' => $contractor->Symbol ]),
            ];

            return $this->returnSuccess($contractor, 'Pomyślnie dodano nowego kontrahenta', 201, $headers);
        } catch (Exception $e) {
            abort(400, $e->getMessage());
        }
    }

    /**
     * Waliduje poprawność przekazanych danych.
     *
     * @param  Request  $request
     * @param  string   $symbol  Symbol kontrahenta służący do wyszukiwania w Subiekcie
     *
     * @return array
     */
    protected function validateContractor(Request $request, string $symbol = ''): array
    {
        $symbol = $symbol ?: $request->json('symbol');
        $name = $request->json('nazwa');
        $firstName = $request->json('imie');
        $lastName = $request->json('nazwisko');
        $nip = $request->json('nip');
        $email = $request->json('email');
        $phoneNo = $request->json('telefon');
        $isCompany = filter_var($request->json('czy_firma'), FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        $city = $request->json('miejscowosc');
        $postalCode = $request->json('kodPocztowy');
        $street = $request->json('ulica');
        $houseNo = $request->json('nrDomu');
        $apartmentNo = $request->json('nrLokalu');

        if (!$symbol) {
            abort(400, 'Identyfikator jest wymagany');
        } elseif ($request->json('czy_firma') === null) {
            abort(400, 'Flaga "czy_firma" jest wymagana');
        } elseif ($isCompany === null) {
            abort(400, 'Flaga "czy_firma" nie jest poprawną wartością logiczną');
        } elseif ($isCompany && !$nip) {
            abort(400, 'NIP jest wymagany dla firm');
        } elseif ($isCompany && !$name) {
            abort(400, 'Nazwa kontrahenta jest wymagana');
        } elseif (!$isCompany && (!$firstName || $lastName)) {
            abort(400, 'Imię i nazwisko kontrahenta są wymagane');
        } elseif (!$email) {
            abort(400, 'E-mail jest wymagany');
        } elseif (!$phoneNo) {
            abort(400, 'Numer telefonu jest wymagany');
        } elseif (!$city) {
            abort(400, 'Miejscowość jest wymagana');
        } elseif (!$postalCode) {
            abort(400, 'Kod pocztowy jest wymagany');
        } elseif (!$street) {
            abort(400, 'Ulica jest wymagana');
        }

        $ret = [
            'symbol' => $symbol,
            'email' => $email,
            'phoneNo' => $phoneNo,
            'isCompany' => $isCompany,
            'city' => $city,
            'postalCode' => $postalCode,
            'street' => $street,
        ];

        if ($houseNo) {
            $ret['houseNo'] = $houseNo;
        }

        if ($apartmentNo) {
            $ret['apartmentNo'] = $apartmentNo;
        }

        if ($isCompany) {
            $ret['name'] = $name;
            $ret['nip'] = $nip;
        } else {
            $ret['firstName'] = $firstName;
            $ret['lastName'] = $lastName;
        }

        return $ret;
    }

    /**
     * Zapisuje zmiany u kontrahenta w Subiekcie.
     * Działa zarówno przy dodawaniu nowego kontrahenta, jak i przy aktualizacji.
     *
     * @param  GT     $contractor  Obiekt kontrahenta w Subiekcie
     * @param  array  $data  Dane kontrahenta do zapisu
     *
     * @return void
     */
    protected function saveContractor($contractor, array $data): void
    {
        $contractor->Symbol = $data['symbol'];

        if ($data['isCompany']) {
            $contractor->NazwaPelna = $data['name'];
            $contractor->Nazwa = Str::substr($data['name'], 0, 100);
            $contractor->NIP = $data['nip'];
            $contractor->Osoba = 0;
        } else {
            $contractor->NazwaPelna = $data['name'];
            $contractor->OsobaImie = $data['name'];
            $contractor->OsobaNazwisko = $data['nip'];
            $contractor->Osoba = 1;
        }

        $contractor->Email = $data['email'];
        $contractor->Miejscowosc = $data['city'];
        $contractor->KodPocztowy = $data['postalCode'];
        $contractor->Ulica = $data['street'];
        $contractor->NrDomu = isset($data['houseNo']) ? $data['houseNo'] : '';
        $contractor->NrLokalu = isset($data['apartmentNo']) ? $data['apartmentNo'] : '';

        if ($contractor->Telefony->Liczba > 0) {
            $phone = $contractor->Telefony[1];
        } else {
            $phone = $contractor->Telefony->Dodaj($data['phoneNo']);
        }

        $phone->Nazwa = 'Numer kontaktowy';
        $phone->Numer = $data['phoneNo'];
        $phone->Typ = 3;

        $contractor->Zapisz();
    }

    /**
     * Zwraca odpowiedź w ustalonym formacie jeśli wszystkie operacje powiodły się.
     *
     * @param  GT      $contractor  Obiekt kontrahenta w Subiekcie
     * @param  string  $message  Wiadomość umieszczana w zwrotce, opcjonalna
     * @param  int     $status  Kod HTTP odpowiedzi, domyślnie 200
     * @param  array   $headers  Nagłówki dołączane do odpowiedzi
     *
     * @return JsonResponse
     */
    protected function returnSuccess(
        $contractor,
        string $message = '',
        int $status = 200,
        array $headers = []
    ): JsonResponse {
        $isCompany = !((bool) $contractor->Osoba);
        $ret = [
            'status' => 'success',
            'message' => $message,
            'data' => [
                'id' => $contractor->Identyfikator,
                'symbol' => $contractor->Symbol,
                'czy_firma' => $isCompany,
            ],
        ];

        if ($isCompany) {
            $ret['data']['nazwaPelna'] = $contractor->NazwaPelna;
            $ret['data']['nazwa'] = $contractor->Nazwa;
            $ret['data']['nip'] = $contractor->NIP;
        } else {
            $ret['data']['imie'] = $contractor->OsobaImie;
            $ret['data']['nazwisko'] = $contractor->OsobaNazwisko;
        }

        $ret['data']['email'] = $contractor->Email;

        if ($contractor->Telefony->Liczba > 0) {
            $ret['data']['telefon'] = $contractor->Telefony[1]->Numer;
        }

        $ret['data']['miejscowosc'] = $contractor->Miejscowosc;
        $ret['data']['kodPocztowy'] = $contractor->KodPocztowy;
        $ret['data']['ulica'] = $contractor->Ulica;
        $ret['data']['nrDomu'] = $contractor->NrDomu;
        $ret['data']['nrLokalu'] = $contractor->NrLokalu;

        if (!$message) {
            unset($ret['message']);
        }

        return response()->json($ret, $status)->withHeaders($headers);
    }

    /**
     * Pobiera kontrahenta z Subiekta.
     *
     * @param  string  $symbol  Symbol kontrahenta służący do wyszukiwania w Subiekcie
     *
     * @return JsonResponse
     * @api
     */
    public function show(string $symbol): JsonResponse
    {
        if (!$symbol || !$this->gt->Kontrahenci->Istnieje($symbol)) {
            abort(404, 'Nie istnieje kontrahent o podanym symbolu');
        }

        try {
            return $this->returnSuccess($this->gt->Kontrahenci->Wczytaj($symbol));
        } catch (Exception $e) {
            abort(400, $e->getMessage());
        }
    }

    /**
     * Aktualizuje kontrahenta w Subiekcie.
     *
     * @param  Request  $request
     * @param  string   $symbol  Symbol kontrahenta służący do wyszukiwania w Subiekcie
     *
     * @return JsonResponse
     * @api
     */
    public function update(Request $request, $symbol): JsonResponse
    {
        $validated = $this->validateContractor($request, $symbol);

        if (!$this->gt->Towary->Kontrahenci($validated['symbol'])) {
            abort(404, 'Nie istnieje kontrahent o podanym symbolu');
        }

        try {
            $contractor = $this->gt->Kontrahenci->Wczytaj($validated['symbol']);
            $this->saveContractor($contractor, $validated);

            return $this->returnSuccess($contractor, 'Pomyślnie zaktualizowano kontrahenta');
        } catch (Exception $e) {
            abort(400, $e->getMessage());
        }
    }
}
